/*jshint esversion: 6 */

define(["src/utils", "src/math/mathUtils", "src/math/Mat3", "lodash"],
function(utils, mathUtils, mat3, lodash) {
"use strict";

var exports = {};

/*============================================================================
	Matrix State
============================================================================*/

/** matrixType enumerates four ways of controlling the Matrix dofs:
 * 0 = 00 = { linear : false, translation : false } = kAffine
 * 1 = 01 = { linear : false, translation : true } = kLinear
 * 2 = 10 = { linear : true, translation : false } = kTranslation
 * 3 = 11 = { linear : true, translation : true } = kNotSet
 */

var	kAffine = 0,
	kLinear = 1,
	kTranslation = 2,
	kNotSet = 3;

var type = {
	kAffine,
	kLinear,
	kTranslation,
	kNotSet,

	merge (a, b) {
		/* jshint bitwise: false */
		return a & b;
	},

	propagate (a) {
		if (a < kTranslation) return kAffine;
		if (a > kTranslation) return kNotSet;

		return kTranslation;
	},

	fromAutomation (auto) {
		var result = 0;
		if (auto.translation) result += 1;
		if (auto.linear) result += 2;
		return result;
	}
};
exports.type = type;

/**
 * Matrix dof is a record describing type (matrixType) and value (mat3).
 */
class Matrix {
    constructor (mdof0) {
        
        lodash.defaults(this, mdof0, {
            // defaults to unset matrix
            type: type.kNotSet,

            // defaults to identity matrix so that it has no effect in matrix operations
            // that way the value can be safely used in all matrix operations (as a noop) even when its not set.
            matrix: mat3()    
        });
    }
}
exports.Matrix = Matrix;

    
/**
 * A List of Matrix dofs (LoM) represents the state of every independent layer *without* sublayers.
 *
 * [Matrix]
 */
var globalTimestamp = 1;
    
class ListOfMatrix {
    constructor (lomObj0) {
        lodash.defaults(this, lomObj0, {
            aDof : [],          // most not be called "value" because of setIn logic that searches for that name in the key path
            aTimestamp : [],    // parallel to aDof
            parent : false
        });
    }
    
    clone () {
        var newLom = new ListOfMatrix({
            aDof : lodash.cloneDeep(this.aDof),
            aTimestamp : lodash.cloneDeep(this.aTimestamp),
            parent : false
        });
        
        return newLom;
    }
    
    touchDof (index) {
        this.aTimestamp[index] = globalTimestamp;
    }   // TODO: hoist timestamp into own class, share with tree; optimize up-touching by shortcutting touch when already equal to globalTimestamp
    
    
    dofChanged (index) {
        this.touchDof(index);
        this.parent.touch();        // tree only care about actual changes (TODO: rename to changed)
    }
    
    hasDofBeenTouchedSinceLastIncrement (index) {
        return (this.aTimestamp[index] > globalTimestamp-1);
    }
    
    getIn (path) { return lodash.get(this.aDof, path); }
    
    getDof (index) { return this.aDof[index]; }

    setMatrix (index, matrix) {
        var dof = this.aDof[index];
                
        if (mat3.equalsApproximately(matrix, dof.matrix)) {
            this.touchDof(index);   // whether or not it changed, still touch
            return;
        }
        
        dof.matrix = matrix;
        
        this.dofChanged(index);
    }
    
    setType (index, type) {
        var dof = this.aDof[index];

        if (dof.type === type) {
            this.touchDof(index);
            return;  // touched, but not changed
        }
        
        dof.type = type;
        
        this.dofChanged(index);
    }
    
    getType (index) {
        return this.aDof[index].type;
    }
    
    setDof (index, dof) {
        this.setMatrix(index, dof.matrix);
        this.setType(index, dof.type);
    }
    
    getNumDofs () {
        return this.aDof.length;
    }

    setIn (path, value) {
        utils.assert(path[0] === "value");
        
        // if this fires, see p4 history for sketch of how to implement, or adjust caller
        utils.assert(path.length > 1, "unimplemented replace-entire-LoM logic in setIn");

        var index = path[1],
            field = path[2];

        switch (field) {
            case "matrix": this.setMatrix(index, value);
                break;

            case "type": this.setType(index, value);
                break;

            case undefined: this.setDof(index, value);
                utils.assert(path.length === 2, "unexpected hole in the path to setIn");
                break;
            default:
                utils.assert(false, "unknown key passed to setIn: " + field);
        }
    }
}
exports.ListOfMatrix = ListOfMatrix;
    
/**
 * Tree<Any>
 */
var cloneTree;
    
    
function privateTouch (tree, ts) {
    
    do {
        // console.logToUser("TOUCH " + tree + " " + ts);
        tree.timestamp = ts;
        tree = tree.parent0;
    } while (tree);
}
    

    
class Tree {
    constructor (treeObj0, aDof0) {      // warning: lom0 will be adopted
        if (aDof0) {
            this.value = new ListOfMatrix({aDof : aDof0, parent : this});
            utils.assert(!treeObj0 || !treeObj0.value, "don't pass both aDof0 and a new value, that's redundant");
        }
        lodash.defaults(this, treeObj0, {
            // list of dofs representing the state of this layer.
            value : new ListOfMatrix({parent : this}),     // empty LoM
            // Map of trees with each entry representing the state of each sub-layer.
            children : {}, 	// maps layerStageID -> Tree,
            timestamp : 0,  // time of last change to any matrix at this level, or below
            parent0 : false // needed so each timestamp change can touch up to the root; root has false parent
        });
        
        this.value.parent = this;   // in case someone passed in a default lom value
    }
    
    getIn (path) {    // FIXME remove all calls to getIn with direct call to lodash.get once we get rid of immutable?
        var result = (path.length ? lodash.get(this, path) : this); // allow empty path to refer to self

        utils.assert(lodash.isUndefined(result) === false, "Tree.getIn(): found no value for path '" + path + "'");

        return result;
    }
    
    
    setIn (path, v) {
        var valueIndex = lodash.lastIndexOf(path, "value");     // the "value" field that holds the LoM
        
        utils.assert(valueIndex !== -1);
        
        var subTree = this.getIn(path.slice(0, valueIndex));
        
        utils.assert(subTree instanceof Tree);
        
        subTree.value.setIn(path.slice(valueIndex), v);          // actual mutation
        
        // TODO: conditionalize the set & touch on the v actually changing, but watch out for direct-touchers
        //          and setters who use the mutated object directly from the data structure
        //          (i.e. see warning in doPostMediateLayer)
        
        return this;
    }
        
    clone () {
        return cloneTree(this);
    }
    
    // timestamp class could be pulled out for sharing outside of Tree usage if we ever need it
        
    static incrementTimestamp () {     // generally called whenever you have an event since-which you want to know about changes (e.g. start of a frame)
                                    
        var compareAgainst = globalTimestamp;
        
        ++globalTimestamp;
        
        return compareAgainst;
    }
    
    touch () {
        privateTouch(this, globalTimestamp);
    }
    
    hasChangedSinceLastIncrement () {
        var bChanged = (this.timestamp > globalTimestamp-1);
        
        /*if (bChanged) {
            console.logToUser(`changed: ${bChanged}, this.timestamp=${this.timestamp}, lastone=${globalTimestamp-1}`);
        }*/
        
        return bChanged;
    }
    
    // if ever want to have more than one event to compare changed-sinceness to, we'll need a hasChangedSince(timestamp) method,
    // which would use the return value from incrementTimestamp
}
exports.Tree = Tree;


cloneTree = function (tree, parent0) {
    var newLom = tree.value.clone(),        // can't use lodash because of parent up-pointer
        newTree = new Tree({
            value : newLom,
            timestamp : tree.timestamp,
            parent0
        }),
        newChildren = newTree.children;
    
    lodash.forOwn(tree.children, function (childTree, layerStageID) {
        newChildren[layerStageID] = cloneTree(childTree, newTree);
    });
    
    return newTree;
};
    

 /**
 * A Tree of Matrix dofs (ToM) represents the state of a layer *with* sublayers.
 * ToM is implemented as a class: 
 *
 * Tree<Matrix>
 */


function isTree (tree) {
	return tree instanceof Tree;
}
exports.isTree = isTree;


var getTreeValue = lodash.property("value"),
	getTreeChildren = lodash.property("children");

// Recursive (deep) update of all 'value' keys in a Tree<X>, 
// while simultaneously walking structurally identical Tree<Y>, Tree<Z>, ...
// Given current values in each tree, the updater function (X Y Z ...) mutates.
function updateValuesWithZipped (updater, first, ...rest) {
    if (lodash.isEmpty(first)) return;
    if (isTree(first)) {
        let tree = first;
        var valRest = rest.map(getTreeValue);
        updater(getTreeValue(tree), ...valRest);

        var forestRest = rest.map(getTreeChildren);
        updateValuesWithZipped(updater, getTreeChildren(tree), ...forestRest);
    } else {
        let forest = first;
        lodash.forEach(forest, function (treeFirst, key) {
            var treeRest = rest.map(f => f[key]);
            updateValuesWithZipped(updater, treeFirst, ...treeRest);
        });
    }
}
Tree.updateValuesWithZipped = updateValuesWithZipped;


/*============================================================================
	Transform State
============================================================================*/

/**
 * A List of Transform dofs (LoT) and Tree of Transform dofs (ToT) offer another representation of Layer state.
 * Although Transform dofs ultimately map back to Matrix dofs, they enable richer composition (i.e. rotation winding)
 * with seperate component for translation, rotation, scale, and so on.
 * 
 * LoT and ToT implementations are also backed by immutable data types:
 * immutable.List<TransformDof> and Tree<TransformDof>.
 *
 * NOTE: Transform is plain JS object, not an immutable type.  Immutable data operations will make shallow copies:
 * *reference* copy not value copy.
 *
 */

/**
 * Transform is plain (mutable) JS object.
 */

var kTransformDefault = {
	type : type.kNotSet,

	x : 0,
	y : 0,
	angle : 0,
	xScale : 1,
	yScale : 1,
	xShear : 0
};
exports.TransformDefault = kTransformDefault;

function Transform (args0) {
	if (args0) {
		lodash.defaults(this, args0, kTransformDefault);
	} else {
		lodash.defaults(this, kTransformDefault);
	}
}
exports.Transform = Transform;

const positionTmp = [],
	scaleTmp = [],
	shearTmp = [],
	rotationTmp = [];

function transformFromMatrix (mdof)
{
	mat3.decomposeAffine(mdof.matrix, positionTmp, scaleTmp, shearTmp, rotationTmp);

	return new Transform({
		type: mdof.type,
		x : positionTmp[0],
		y : positionTmp[1],
		angle : mathUtils.angle(rotationTmp),
		xScale : scaleTmp[0],
		yScale : scaleTmp[1],
		xShear : shearTmp[0]
	});
}
exports.transformFromMatrix = transformFromMatrix;

function matFromTransform (tdof, result0) {
	positionTmp[0] = tdof.x; 
	positionTmp[1] = tdof.y;

	scaleTmp[0] = tdof.xScale; 
	scaleTmp[1] = tdof.yScale;

	shearTmp[0] = tdof.xShear;

	return mat3.affine(positionTmp, scaleTmp, shearTmp, tdof.angle, result0);
}
exports.matFromTransform = matFromTransform;

function matrixFromTransform (tdof) 
{
	return new Matrix({
		type : tdof.type,
		matrix : matFromTransform(tdof)
	});
}
exports.matrixFromTransform = matrixFromTransform;

function typeFromKey (key) {
	if (key === "x" || key === "y")	return type.kTranslation;

	return type.kLinear;
}
exports.typeFromKey = typeFromKey;

return exports;
});  // end define